// PDF转图片处理脚本 - 用于明细上传页面
// 当用户选择PDF文件时，自动转换为图片后上传
(function() {
    'use strict';

    // 获取主文档（如果不在iframe中，返回当前文档）
    function getMainDocument() {
        try {
            if (window.top && window.top !== window.self) {
                return window.top.document;
            }
            return document;
        } catch (e) {
            console.log('[PDF转图片] 无法访问主文档，使用当前文档:', e.message);
            return document;
        }
    }

    // 检测是否是明细上传页面
    function isDetailUploadPage() {
        try {
            // 获取主文档（tab在主文档中）
            const mainDoc = getMainDocument();
            
            // 查找当前选中的 tab
            const selectedTab = mainDoc.querySelector('.tabs-selected');
            if (!selectedTab) {
                console.log('[PDF转图片] 未找到选中的 tab');
                return false;
            }
            
            // 查找选中 tab 中的标题元素
            const tabTitle = selectedTab.querySelector('.tabs-title');
            if (!tabTitle) {
                console.log('[PDF转图片] 未找到 tab 标题');
                return false;
            }
            
            // 获取标题文本
            const titleText = tabTitle.textContent.trim();
            console.log('[PDF转图片] 当前 tab 标题:', titleText);
            
            // 检查标题是否包含"创建明细"或"编辑明细"
            if (titleText.includes('新增明细') || titleText.includes('编辑明细')) {
                console.log('[PDF转图片] 检测到明细上传页面');
                return true;
            }
            
            return false;
        } catch (error) {
            console.error('[PDF转图片] 检测页面类型时出错:', error);
            return false;
        }
    }

    // 初始化PDF.js库（已在manifest.json中加载）
    function loadPDFJS() {
        return new Promise((resolve, reject) => {
            // PDF.js应该在manifest.json中已经作为content script加载了
            // 检查是否已经加载
            if (window.pdfjsLib) {
                // 设置worker路径（如果还没设置）
                if (!window.pdfjsLib.GlobalWorkerOptions.workerSrc) {
                    if (typeof chrome !== 'undefined' && chrome.runtime && chrome.runtime.getURL) {
                        window.pdfjsLib.GlobalWorkerOptions.workerSrc = chrome.runtime.getURL('lib/pdf.worker.min.js');
                    }
                }
                resolve(window.pdfjsLib);
                return;
            }

            // 如果PDF.js还没加载，等待一下（可能是异步加载延迟）
            let retries = 0;
            const maxRetries = 20; // 增加重试次数
            const checkInterval = setInterval(() => {
                retries++;
                
                // 尝试多种可能的导出方式
                let pdfjsLib = window.pdfjsLib || 
                              (typeof globalThis !== 'undefined' && globalThis.pdfjsLib) ||
                              (typeof self !== 'undefined' && self.pdfjsLib) ||
                              window["pdfjs-dist/build/pdf"] ||
                              (typeof globalThis !== 'undefined' && globalThis["pdfjs-dist/build/pdf"]) ||
                              (typeof self !== 'undefined' && self["pdfjs-dist/build/pdf"]);
                
                if (pdfjsLib) {
                    clearInterval(checkInterval);
                    console.log('[PDF转图片] PDF.js库已找到，设置Worker路径...');
                    // 设置worker路径
                    if (!pdfjsLib.GlobalWorkerOptions.workerSrc) {
                        if (typeof chrome !== 'undefined' && chrome.runtime && chrome.runtime.getURL) {
                            pdfjsLib.GlobalWorkerOptions.workerSrc = chrome.runtime.getURL('lib/pdf.worker.min.js');
                        }
                    }
                    resolve(pdfjsLib);
                } else if (retries >= maxRetries) {
                    clearInterval(checkInterval);
                    console.error('[PDF转图片] PDF.js加载失败：库未在manifest.json中正确加载');
                    const debugInfo = {
                        'window.pdfjsLib': window.pdfjsLib,
                        'globalThis.pdfjsLib': typeof globalThis !== 'undefined' ? globalThis.pdfjsLib : undefined,
                        'self.pdfjsLib': typeof self !== 'undefined' ? self.pdfjsLib : undefined,
                        'window["pdfjs-dist/build/pdf"]': window["pdfjs-dist/build/pdf"]
                    };
                    console.error('[PDF转图片] 已检查的全局对象:', debugInfo);
                    reject(new Error('PDF.js库未加载，请检查manifest.json配置'));
                }
            }, 100); // 每100ms检查一次
        });
    }

    // 压缩图片，确保文件大小不超过指定限制，同时保持清晰度
    /**
     * 压缩图片，确保文件大小不超过指定限制
     * @param {Blob} imageBlob - 原始图片blob
     * @param {number} maxSizeKB - 最大文件大小（KB），默认3000KB（3MB）
     * @param {number} maxDimension - 最大尺寸（宽度或高度），默认2000px
     * @returns {Promise<Blob>} 压缩后的图片blob
     */
    async function compressImage(imageBlob, maxSizeKB = 3000, maxDimension = 2000) {
        return new Promise((resolve) => {
            const img = new Image();
            img.onload = () => {
                // 计算是否需要缩放
                let width = img.width;
                let height = img.height;
                let needsResize = false;
                
                if (width > maxDimension || height > maxDimension) {
                    needsResize = true;
                    const ratio = Math.min(maxDimension / width, maxDimension / height);
                    width = Math.floor(width * ratio);
                    height = Math.floor(height * ratio);
                    console.log(`[PDF转图片] 图片尺寸过大，缩放至 ${width}x${height}`);
                }
                
                // 创建canvas并绘制图片
                const canvas = document.createElement('canvas');
                const ctx = canvas.getContext('2d');
                canvas.width = width;
                canvas.height = height;
                ctx.drawImage(img, 0, 0, width, height);
                
                // 尝试不同的质量级别，直到满足大小要求
                const tryCompress = (quality) => {
                    canvas.toBlob((compressedBlob) => {
                        const sizeKB = compressedBlob.size / 1024;
                        console.log(`[PDF转图片] 压缩后大小: ${sizeKB.toFixed(2)}KB，质量: ${(quality * 100).toFixed(0)}%`);
                        
                        if (sizeKB <= maxSizeKB || quality <= 0.6) {
                            // 如果达到目标大小或质量已降至60%，停止压缩
                            resolve(compressedBlob);
                        } else {
                            // 质量降低0.05，继续尝试
                            tryCompress(Math.max(0.6, quality - 0.05));
                        }
                    }, 'image/jpeg', quality);
                };
                
                // 从0.9质量开始尝试，保证清晰度
                tryCompress(0.9);
            };
            img.onerror = () => {
                console.warn('[PDF转图片] 图片加载失败，返回原始blob');
                resolve(imageBlob); // 如果加载失败，返回原blob
            };
            img.src = URL.createObjectURL(imageBlob);
        });
    }

    // 显示全屏查看器
    async function showFullscreenViewer(pageNum, pdf, pdfjsLib, mainDoc) {
        try {
            // 创建全屏遮罩
            const fullscreenOverlay = mainDoc.createElement('div');
            fullscreenOverlay.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0, 0, 0, 0.95);
                z-index: 20000;
                display: flex;
                align-items: center;
                justify-content: center;
                cursor: pointer;
            `;
            
            // 创建容器
            const container = mainDoc.createElement('div');
            container.style.cssText = `
                position: relative;
                max-width: 95%;
                max-height: 95%;
                display: flex;
                flex-direction: column;
                align-items: center;
            `;
            
            // 创建标题栏
            const header = mainDoc.createElement('div');
            header.style.cssText = `
                color: white;
                padding: 15px;
                text-align: center;
                font-size: 18px;
                font-weight: bold;
                margin-bottom: 10px;
            `;
            header.textContent = `第 ${pageNum} 页 / 共 ${pdf.numPages} 页`;
            
            // 创建加载提示
            const loadingDiv = mainDoc.createElement('div');
            loadingDiv.style.cssText = `
                color: white;
                padding: 20px;
                text-align: center;
                font-size: 16px;
            `;
            loadingDiv.textContent = '正在加载高清图片...';
            
            // 创建图片容器
            const imageContainer = mainDoc.createElement('div');
            imageContainer.style.cssText = `
                position: relative;
                max-width: 100%;
                max-height: 80vh;
                overflow: auto;
                display: flex;
                align-items: center;
                justify-content: center;
                background: #1a1a1a;
                border-radius: 4px;
            `;
            
            const fullscreenImage = mainDoc.createElement('img');
            fullscreenImage.style.cssText = `
                max-width: 100%;
                max-height: 80vh;
                display: block;
                margin: auto;
            `;
            
            // 创建关闭按钮
            const closeBtn = mainDoc.createElement('button');
            closeBtn.textContent = '×';
            closeBtn.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: rgba(255, 255, 255, 0.2);
                color: white;
                border: 2px solid rgba(255, 255, 255, 0.5);
                border-radius: 50%;
                width: 50px;
                height: 50px;
                font-size: 32px;
                cursor: pointer;
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 20001;
                transition: all 0.2s;
            `;
            closeBtn.addEventListener('mouseenter', () => {
                closeBtn.style.background = 'rgba(255, 255, 255, 0.3)';
                closeBtn.style.transform = 'scale(1.1)';
            });
            closeBtn.addEventListener('mouseleave', () => {
                closeBtn.style.background = 'rgba(255, 255, 255, 0.2)';
                closeBtn.style.transform = 'scale(1)';
            });
            
            const close = () => {
                mainDoc.body.removeChild(fullscreenOverlay);
                if (closeBtn.parentNode) {
                    mainDoc.body.removeChild(closeBtn);
                }
            };
            
            closeBtn.onclick = (e) => {
                e.stopPropagation();
                close();
            };
            
            fullscreenOverlay.onclick = (e) => {
                if (e.target === fullscreenOverlay) {
                    close();
                }
            };
            
            // 组装元素
            container.appendChild(header);
            container.appendChild(loadingDiv);
            imageContainer.appendChild(fullscreenImage);
            container.appendChild(imageContainer);
            fullscreenOverlay.appendChild(container);
            mainDoc.body.appendChild(fullscreenOverlay);
            mainDoc.body.appendChild(closeBtn);
            
            // 加载高清版本的页面
            try {
                const page = await pdf.getPage(pageNum);
                // 使用更高的缩放比例以获得清晰的大图
                const viewport = page.getViewport({ scale: 3.0 });
                
                const canvas = document.createElement('canvas');
                const context = canvas.getContext('2d');
                canvas.height = viewport.height;
                canvas.width = viewport.width;
                
                await page.render({
                    canvasContext: context,
                    viewport: viewport
                }).promise;
                
                // 设置图片
                fullscreenImage.src = canvas.toDataURL('image/png', 1.0);
                fullscreenImage.onload = () => {
                    loadingDiv.remove();
                };
                fullscreenImage.onerror = () => {
                    loadingDiv.textContent = '图片加载失败';
                    loadingDiv.style.color = '#f44336';
                };
            } catch (error) {
                console.error(`[PDF转图片] 加载第 ${pageNum} 页高清版本失败:`, error);
                loadingDiv.textContent = '加载失败: ' + error.message;
                loadingDiv.style.color = '#f44336';
            }
            
            // 支持ESC键关闭
            const escHandler = (e) => {
                if (e.key === 'Escape' && mainDoc.body.contains(fullscreenOverlay)) {
                    close();
                    // 从正确的文档移除监听器
                    const docToRemove = mainDoc || document;
                    docToRemove.removeEventListener('keydown', escHandler);
                }
            };
            // 在正确的文档上添加监听器
            const docToAdd = mainDoc || document;
            docToAdd.addEventListener('keydown', escHandler);
            
        } catch (error) {
            console.error('[PDF转图片] 创建全屏查看器失败:', error);
        }
    }

    // 使用原生DOM API创建页面选择对话框
    async function createNativePageSelectionDialog(pdfFile, pdf, pdfjsLib) {
        return new Promise((resolve, reject) => {
            try {
                const mainWindow = window.top || window;
                const mainDoc = mainWindow.document || document;
                
                // 创建模态遮罩
                const overlay = mainDoc.createElement('div');
                overlay.style.cssText = `
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: rgba(0, 0, 0, 0.5);
                    z-index: 10000;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                `;
                
                // 创建对话框容器
                const dialog = mainDoc.createElement('div');
                dialog.style.cssText = `
                    background: white;
                    border-radius: 8px;
                    padding: 0;
                    max-width: 900px;
                    max-height: 80vh;
                    width: 90%;
                    display: flex;
                    flex-direction: column;
                    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
                `;
                
                // 创建标题栏
                const titleBar = mainDoc.createElement('div');
                titleBar.style.cssText = `
                    padding: 15px 20px;
                    border-bottom: 1px solid #ddd;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                    background: #f5f5f5;
                    border-radius: 8px 8px 0 0;
                `;
                const title = mainDoc.createElement('h3');
                title.textContent = `选择要转换的页面 - ${pdfFile.name}`;
                title.style.cssText = 'margin: 0; font-size: 16px; font-weight: bold;';
                
                const closeBtn = mainDoc.createElement('button');
                closeBtn.textContent = '×';
                closeBtn.style.cssText = `
                    background: none;
                    border: none;
                    font-size: 24px;
                    cursor: pointer;
                    padding: 0;
                    width: 30px;
                    height: 30px;
                    line-height: 1;
                    color: #666;
                `;
                closeBtn.onclick = () => {
                    mainDoc.body.removeChild(overlay);
                    reject(new Error('用户取消'));
                };
                
                titleBar.appendChild(title);
                titleBar.appendChild(closeBtn);
                
                // 创建控制栏
                const controls = mainDoc.createElement('div');
                controls.style.cssText = `
                    padding: 15px;
                    border-bottom: 1px solid #ddd;
                    display: flex;
                    gap: 10px;
                    align-items: center;
                    background: #f9f9f9;
                `;
                
                const selectAllBtn = mainDoc.createElement('button');
                selectAllBtn.textContent = '全选';
                selectAllBtn.style.cssText = `
                    padding: 5px 15px;
                    background: #4CAF50;
                    color: white;
                    border: none;
                    border-radius: 3px;
                    cursor: pointer;
                `;
                
                const selectNoneBtn = mainDoc.createElement('button');
                selectNoneBtn.textContent = '全不选';
                selectNoneBtn.style.cssText = `
                    padding: 5px 15px;
                    background: #f44336;
                    color: white;
                    border: none;
                    border-radius: 3px;
                    cursor: pointer;
                `;
                
                const pageCountSpan = mainDoc.createElement('span');
                pageCountSpan.style.cssText = 'margin-left: auto; font-weight: bold;';
                
                controls.appendChild(selectAllBtn);
                controls.appendChild(selectNoneBtn);
                controls.appendChild(pageCountSpan);
                
                // 创建内容区域
                const content = mainDoc.createElement('div');
                content.style.cssText = `
                    padding: 15px;
                    overflow-y: auto;
                    flex: 1;
                    display: grid;
                    grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
                    gap: 15px;
                `;
                
                // 创建按钮栏
                const buttonBar = mainDoc.createElement('div');
                buttonBar.style.cssText = `
                    padding: 15px;
                    border-top: 1px solid #ddd;
                    display: flex;
                    justify-content: flex-end;
                    gap: 10px;
                    background: #f5f5f5;
                    border-radius: 0 0 8px 8px;
                `;
                
                const okBtn = mainDoc.createElement('button');
                okBtn.textContent = '确定';
                okBtn.style.cssText = `
                    padding: 8px 20px;
                    background: #2196F3;
                    color: white;
                    border: none;
                    border-radius: 3px;
                    cursor: pointer;
                    font-size: 14px;
                `;
                
                const cancelBtn = mainDoc.createElement('button');
                cancelBtn.textContent = '取消';
                cancelBtn.style.cssText = `
                    padding: 8px 20px;
                    background: #ccc;
                    color: #333;
                    border: none;
                    border-radius: 3px;
                    cursor: pointer;
                    font-size: 14px;
                `;
                
                cancelBtn.onclick = () => {
                    mainDoc.body.removeChild(overlay);
                    reject(new Error('用户取消'));
                };
                
                buttonBar.appendChild(cancelBtn);
                buttonBar.appendChild(okBtn);
                
                // 组装对话框
                dialog.appendChild(titleBar);
                dialog.appendChild(controls);
                dialog.appendChild(content);
                dialog.appendChild(buttonBar);
                overlay.appendChild(dialog);
                mainDoc.body.appendChild(overlay);
                
                const pageCheckboxes = [];
                let selectedPages = [];
                
                // 更新选中页面数量
                const updatePageCount = () => {
                    selectedPages = pageCheckboxes
                        .filter(cb => cb.checked)
                        .map(cb => parseInt(cb.value))
                        .sort((a, b) => a - b);
                    pageCountSpan.textContent = `已选择 ${selectedPages.length}/${pdf.numPages} 页`;
                };
                
                // 生成页面缩略图
                (async () => {
                    showLoading('正在加载PDF页面预览...');
                    
                    for (let pageNum = 1; pageNum <= pdf.numPages; pageNum++) {
                        try {
                            const page = await pdf.getPage(pageNum);
                            const viewport = page.getViewport({ scale: 0.5 });
                            
                            const canvas = document.createElement('canvas');
                            const context = canvas.getContext('2d');
                            canvas.height = viewport.height;
                            canvas.width = viewport.width;
                            
                            await page.render({
                                canvasContext: context,
                                viewport: viewport
                            }).promise;
                            
                            // 创建页面项
                            const pageItem = mainDoc.createElement('div');
                            pageItem.style.cssText = `
                                border: 2px solid #ddd;
                                border-radius: 4px;
                                padding: 10px;
                                background: white;
                                cursor: pointer;
                                transition: all 0.2s;
                                position: relative;
                            `;
                            
                            // 创建全屏按钮
                            const fullscreenBtn = mainDoc.createElement('button');
                            fullscreenBtn.innerHTML = '🔍';
                            fullscreenBtn.title = '全屏查看';
                            fullscreenBtn.style.cssText = `
                                position: absolute;
                                top: 5px;
                                right: 5px;
                                background: transparent;
                                color: white;
                                border: none;
                                border-radius: 4px;
                                width: 30px;
                                height: 30px;
                                cursor: pointer;
                                font-size: 16px;
                                display: flex;
                                align-items: center;
                                justify-content: center;
                                z-index: 10;
                                transition: all 0.2s;
                            `;
                            fullscreenBtn.addEventListener('mouseenter', () => {
                                fullscreenBtn.style.background = 'transparent';
                                fullscreenBtn.style.transform = 'scale(1.1)';
                            });
                            fullscreenBtn.addEventListener('mouseleave', () => {
                                fullscreenBtn.style.background = 'transparent';
                                fullscreenBtn.style.transform = 'scale(1)';
                            });
                            fullscreenBtn.addEventListener('click', (e) => {
                                e.stopPropagation(); // 阻止触发页面项的点击事件
                                showFullscreenViewer(pageNum, pdf, pdfjsLib, mainDoc);
                            });
                            
                            const checkbox = mainDoc.createElement('input');
                            checkbox.type = 'checkbox';
                            checkbox.value = pageNum;
                            checkbox.checked = true;
                            checkbox.style.cssText = `
                                margin-right: 8px;
                                cursor: pointer;
                                width: 18px;
                                height: 18px;
                            `;
                            
                            const label = mainDoc.createElement('label');
                            label.style.cssText = `
                                display: flex;
                                align-items: center;
                                margin-bottom: 8px;
                                font-weight: bold;
                                cursor: pointer;
                            `;
                            label.textContent = `第 ${pageNum} 页`;
                            label.insertBefore(checkbox, label.firstChild);
                            
                            const thumbnail = mainDoc.createElement('img');
                            thumbnail.src = canvas.toDataURL('image/png');
                            thumbnail.style.cssText = `
                                width: 100%;
                                height: auto;
                                border: 1px solid #eee;
                                border-radius: 2px;
                                display: block;
                            `;
                            
                            // 事件处理
                            checkbox.addEventListener('change', () => {
                                updatePageCount();
                                pageItem.style.borderColor = checkbox.checked ? '#4CAF50' : '#ddd';
                                pageItem.style.backgroundColor = checkbox.checked ? '#f1f8f4' : 'white';
                            });
                            
                            pageItem.addEventListener('click', (e) => {
                                // 如果点击的是全屏按钮或复选框，不切换选中状态
                                if (e.target === checkbox || e.target === fullscreenBtn || e.target.closest('button')) {
                                    return;
                                }
                                checkbox.checked = !checkbox.checked;
                                checkbox.dispatchEvent(new Event('change'));
                            });
                            
                            pageItem.addEventListener('mouseenter', () => {
                                pageItem.style.borderColor = '#4CAF50';
                            });
                            
                            pageItem.addEventListener('mouseleave', () => {
                                if (!checkbox.checked) {
                                    pageItem.style.borderColor = '#ddd';
                                }
                            });
                            
                            pageItem.appendChild(fullscreenBtn);
                            pageItem.appendChild(label);
                            pageItem.appendChild(thumbnail);
                            content.appendChild(pageItem);
                            pageCheckboxes.push(checkbox);
                            
                            // 初始状态
                            pageItem.style.borderColor = '#4CAF50';
                            pageItem.style.backgroundColor = '#f1f8f4';
                            
                            if (pageNum % 5 === 0 || pageNum === pdf.numPages) {
                                showLoading(`正在加载PDF页面预览（${pageNum}/${pdf.numPages}）...`);
                            }
                        } catch (error) {
                            console.error(`[PDF转图片] 加载第 ${pageNum} 页预览失败:`, error);
                        }
                    }
                    
                    hideLoading();
                    updatePageCount();
                })();
                
                // 全选/全不选
                selectAllBtn.onclick = () => {
                    pageCheckboxes.forEach(cb => {
                        cb.checked = true;
                        const item = cb.closest('div');
                        if (item) {
                            item.style.borderColor = '#4CAF50';
                            item.style.backgroundColor = '#f1f8f4';
                        }
                    });
                    updatePageCount();
                };
                
                selectNoneBtn.onclick = () => {
                    pageCheckboxes.forEach(cb => {
                        cb.checked = false;
                        const item = cb.closest('div');
                        if (item) {
                            item.style.borderColor = '#ddd';
                            item.style.backgroundColor = 'white';
                        }
                    });
                    updatePageCount();
                };
                
                // 确定按钮
                okBtn.onclick = () => {
                    if (selectedPages.length === 0) {
                        alert('请至少选择一页');
                        return;
                    }
                    mainDoc.body.removeChild(overlay);
                    resolve(selectedPages);
                };
                
                // 点击遮罩关闭
                overlay.addEventListener('click', (e) => {
                    if (e.target === overlay) {
                        mainDoc.body.removeChild(overlay);
                        reject(new Error('用户取消'));
                    }
                });
                
            } catch (error) {
                console.error('[PDF转图片] 创建原生对话框失败:', error);
                reject(error);
            }
        });
    }

    // 显示页面选择对话框（直接使用原生DOM API）
    async function showPageSelectionDialog(pdfFile, pdf, pdfjsLib) {
        return createNativePageSelectionDialog(pdfFile, pdf, pdfjsLib);
    }

    // 将PDF文件转换为图片
    async function convertPDFToImages(pdfFile, onProgress, selectedPages = null) {
        try {
            // 加载PDF.js
            const pdfjsLib = await loadPDFJS();
            
            // 读取PDF文件
            const arrayBuffer = await pdfFile.arrayBuffer();
            const pdf = await pdfjsLib.getDocument({ data: arrayBuffer }).promise;
            
            console.log(`[PDF转图片] PDF文件有 ${pdf.numPages} 页`);
            
            // 如果没有指定要转换的页面，显示选择对话框
            let pagesToConvert = selectedPages;
            if (!pagesToConvert) {
                try {
                    pagesToConvert = await showPageSelectionDialog(pdfFile, pdf, pdfjsLib);
                    console.log(`[PDF转图片] 用户选择了页面: ${pagesToConvert.join(', ')}`);
                } catch (error) {
                    if (error.message === '用户取消') {
                        throw error; // 用户取消，直接抛出
                    }
                    console.warn('[PDF转图片] 页面选择对话框出错，转换所有页面:', error);
                    // 如果对话框出错，默认转换所有页面
                    pagesToConvert = Array.from({ length: pdf.numPages }, (_, i) => i + 1);
                }
            }
            
            const imageFiles = [];
            const baseFileName = pdfFile.name.replace(/\.pdf$/i, '');
            
            // 遍历选中的页面
            for (let i = 0; i < pagesToConvert.length; i++) {
                const pageNum = pagesToConvert[i];
                const currentIndex = i + 1;
                const totalSelected = pagesToConvert.length;
                
                // 更新进度提示
                if (onProgress) {
                    onProgress(currentIndex, totalSelected, pageNum);
                }
                
                const page = await pdf.getPage(pageNum);
                
                // 先获取页面基础尺寸，根据页面大小自适应缩放比例
                const baseViewport = page.getViewport({ scale: 1.0 });
                const pageWidth = baseViewport.width;
                const pageHeight = baseViewport.height;
                
                // 根据页面尺寸自适应缩放比例，平衡清晰度和文件大小
                let scale = 2.0; // 默认2倍，适合标准页面
                if (pageWidth > 1200 || pageHeight > 1600) {
                    scale = 1.5; // 大页面使用较低缩放，避免文件过大
                    console.log(`[PDF转图片] 页面较大(${pageWidth.toFixed(0)}x${pageHeight.toFixed(0)})，使用缩放比例 ${scale}`);
                } else if (pageWidth < 600 && pageHeight < 800) {
                    scale = 2.5; // 小页面可以使用更高缩放
                    console.log(`[PDF转图片] 页面较小(${pageWidth.toFixed(0)}x${pageHeight.toFixed(0)})，使用缩放比例 ${scale}`);
                }
                
                const viewport = page.getViewport({ scale: scale });
                
                // 创建canvas
                const canvas = document.createElement('canvas');
                const context = canvas.getContext('2d');
                canvas.height = viewport.height;
                canvas.width = viewport.width;
                
                // 渲染PDF页面到canvas
                await page.render({
                    canvasContext: context,
                    viewport: viewport
                }).promise;
                
                // 先将canvas转换为PNG（无损格式，保证初始清晰度）
                let blob = await new Promise((resolve) => {
                    canvas.toBlob((blob) => {
                        resolve(blob);
                    }, 'image/png', 0.95);
                });
                
                const sizeKB = blob.size / 1024;
                console.log(`[PDF转图片] 第 ${pageNum} 页PNG转换后大小: ${sizeKB.toFixed(2)}KB`);
                
                let fileName;
                // 如果文件大于3MB，进行压缩并转为JPEG
                if (sizeKB > 3000) {
                    console.log(`[PDF转图片] 文件较大(${sizeKB.toFixed(2)}KB)，开始压缩为JPEG...`);
                    blob = await compressImage(blob, 3000, 2000);
                    const compressedSizeKB = blob.size / 1024;
                    console.log(`[PDF转图片] 压缩完成，最终大小: ${compressedSizeKB.toFixed(2)}KB`);
                    
                    // 使用JPEG扩展名
                    fileName = pagesToConvert.length > 1 
                        ? `${baseFileName}_${pageNum}.jpg` 
                        : `${baseFileName}.jpg`;
                    
                    // 创建File对象（JPEG格式）
                    const imageFile = new File([blob], fileName, { type: 'image/jpeg' });
                    imageFiles.push(imageFile);
                } else {
                    // 文件不大，保持PNG格式
                    fileName = pagesToConvert.length > 1 
                        ? `${baseFileName}_${pageNum}.png` 
                        : `${baseFileName}.png`;
                    
                    // 创建File对象（PNG格式）
                    const imageFile = new File([blob], fileName, { type: 'image/png' });
                    imageFiles.push(imageFile);
                }
                
                console.log(`[PDF转图片] 第 ${pageNum} 页已转换为图片: ${fileName}`);
            }
            
            return imageFiles;
        } catch (error) {
            console.error('[PDF转图片] 转换失败:', error);
            throw error;
        }
    }

    // 检测文件是否为PDF
    function isPDFFile(file) {
        return file.type === 'application/pdf' || 
               file.name.toLowerCase().endsWith('.pdf');
    }

    // 显示加载提示
    function showLoading(message) {
        try {
            // 尝试在主文档和当前文档中查找loading2元素
            const mainDoc = getMainDocument();
            let loadingDiv = mainDoc.getElementById('loading2');
            
            // 如果主文档中没找到，尝试当前文档
            if (!loadingDiv) {
                loadingDiv = document.getElementById('loading2');
            }
            
            if (loadingDiv) {
                loadingDiv.textContent = message || '正在处理PDF，请稍等...';
                loadingDiv.style.display = 'block';
                console.log('[PDF转图片] 显示加载提示:', message);
            } else {
                console.warn('[PDF转图片] 未找到loading2元素，尝试创建临时提示');
                // 如果找不到loading2，使用jQuery dialog显示进度
                const mainWindow = window.top || window;
                if (mainWindow.$ && typeof mainWindow.$ === 'function') {
                    // 确保对话框在主文档中创建
                    const mainDoc = mainWindow.document || document;
                    const body = mainDoc.body || document.body;
                    
                    // 创建一个模态对话框显示进度
                    if (!mainWindow._pdfLoadingDialog) {
                        mainWindow._pdfLoadingDialog = mainWindow.$('<div style="padding: 20px; font-size: 14px;"></div>')
                            .html(message || '正在处理PDF，请稍等...')
                            .appendTo(body);
                        mainWindow._pdfLoadingDialog.dialog({
                            title: 'PDF转换中',
                            modal: true,
                            closable: false,
                            resizable: false,
                            width: 400,
                            height: 150,
                            zIndex: 10002
                        });
                        console.log('[PDF转图片] 已创建对话框提示');
                    } else {
                        // 更新对话框内容
                        mainWindow._pdfLoadingDialog.html(message || '正在处理PDF，请稍等...');
                        if (!mainWindow._pdfLoadingDialog.dialog('isOpen')) {
                            mainWindow._pdfLoadingDialog.dialog('open');
                        }
                    }
                } else if (window.$ && typeof window.$ === 'function') {
                    // 如果在当前窗口中有jQuery，也在当前窗口创建
                    if (!window._pdfLoadingDialog) {
                        window._pdfLoadingDialog = window.$('<div style="padding: 20px; font-size: 14px;"></div>')
                            .html(message || '正在处理PDF，请稍等...')
                            .appendTo(document.body);
                        window._pdfLoadingDialog.dialog({
                            title: 'PDF转换中',
                            modal: true,
                            closable: false,
                            resizable: false,
                            width: 400,
                            height: 150,
                            zIndex: 10002
                        });
                        console.log('[PDF转图片] 已创建对话框提示（当前窗口）');
                    } else {
                        window._pdfLoadingDialog.html(message || '正在处理PDF，请稍等...');
                        if (!window._pdfLoadingDialog.dialog('isOpen')) {
                            window._pdfLoadingDialog.dialog('open');
                        }
                    }
                } else {
                    // 最后的备选方案：控制台日志（至少开发者能看到）
                    console.log('[PDF转图片]', message || '正在处理PDF，请稍等...');
                }
            }
        } catch (error) {
            console.error('[PDF转图片] 显示加载提示时出错:', error);
        }
    }

    // 隐藏加载提示
    function hideLoading() {
        try {
            // 尝试在主文档和当前文档中查找loading2元素
            const mainDoc = getMainDocument();
            let loadingDiv = mainDoc.getElementById('loading2');
            
            // 如果主文档中没找到，尝试当前文档
            if (!loadingDiv) {
                loadingDiv = document.getElementById('loading2');
            }
            
            if (loadingDiv) {
                loadingDiv.style.display = 'none';
                console.log('[PDF转图片] 隐藏加载提示');
            }
            
            // 如果使用了messager对话框，关闭它
            const mainWindow = window.top || window;
            if (mainWindow._pdfLoadingDialog) {
                try {
                    mainWindow._pdfLoadingDialog.dialog('close');
                    mainWindow._pdfLoadingDialog.remove();
                    mainWindow._pdfLoadingDialog = null;
                } catch (e) {
                    console.warn('[PDF转图片] 关闭对话框时出错:', e);
                }
            }
            if (window._pdfLoadingDialog && window !== mainWindow) {
                try {
                    window._pdfLoadingDialog.dialog('close');
                    window._pdfLoadingDialog.remove();
                    window._pdfLoadingDialog = null;
                } catch (e) {
                    console.warn('[PDF转图片] 关闭对话框时出错:', e);
                }
            }
        } catch (error) {
            console.error('[PDF转图片] 隐藏加载提示时出错:', error);
        }
    }

    // 等待上传完成
    function waitForUploadCompletion(expectedImageCount, fileInput, providedInitialFileCount) {
        console.log(`[PDF转图片] 开始监听上传完成，期望 ${expectedImageCount} 张图片`);
        
        // 查找文件列表容器的辅助函数（在所有可能的frames中查找）
        const findFlistContainer = () => {
            // 先尝试主文档
            const mainDoc = getMainDocument();
            let flistContainer = mainDoc.querySelector('#easyContainer .flist');
            if (flistContainer) {
                console.log('[PDF转图片] 在主文档中找到文件列表容器');
                return flistContainer;
            }
            
            // 尝试当前文档
            flistContainer = document.querySelector('#easyContainer .flist');
            if (flistContainer) {
                console.log('[PDF转图片] 在当前文档中找到文件列表容器');
                return flistContainer;
            }
            
            // 尝试在所有frames中查找
            try {
                const frames = window.frames;
                for (let i = 0; i < frames.length; i++) {
                    try {
                        const frameDoc = frames[i].document;
                        if (frameDoc) {
                            flistContainer = frameDoc.querySelector('#easyContainer .flist');
                            if (flistContainer) {
                                console.log(`[PDF转图片] 在frame[${i}]中找到文件列表容器`);
                                return flistContainer;
                            }
                        }
                    } catch (e) {
                        // 跨域frame无法访问，忽略
                    }
                }
            } catch (e) {
                console.warn('[PDF转图片] 无法访问frames:', e);
            }
            
            return null;
        };
        
        // 查找文件列表容器
        let flistContainer = findFlistContainer();
        
        if (!flistContainer) {
            console.warn('[PDF转图片] 未找到文件列表容器，将延迟60秒后关闭提示');
            // 如果找不到文件列表，延迟60秒后关闭（给上传足够时间）
            setTimeout(() => {
                hideLoading();
                showUploadSuccessMessage(expectedImageCount);
            }, 60000);
            return;
        }
        
        // 使用传入的初始文件数量，如果没有则重新计算
        let initialFileCount = providedInitialFileCount;
        if (typeof initialFileCount !== 'number') {
            initialFileCount = flistContainer.querySelectorAll('.flist-item').length;
            console.warn(`[PDF转图片] 未提供初始文件数量，重新计算得到: ${initialFileCount}`);
        }
        console.log(`[PDF转图片] 初始文件数量: ${initialFileCount}，期望新增: ${expectedImageCount}`);
        
        // 设置超时保护：如果60秒内还没上传完成，也关闭提示
        let timeoutId;
        let checkIntervalId;
        let isCompleted = false;
        let observer = null;
        
        // 获取转换后的图片文件名（用于匹配）
        const expectedFileNames = [];
        try {
            // 从fileInput获取转换后的文件
            if (fileInput && fileInput.files) {
                Array.from(fileInput.files).forEach(file => {
                    // 只检查PNG文件（转换后的图片）
                    if (file.name.endsWith('.png') || file.name.endsWith('.PNG')) {
                        expectedFileNames.push(file.name);
                    }
                });
            }
        } catch (e) {
            console.warn('[PDF转图片] 无法获取预期文件名:', e);
        }
        console.log(`[PDF转图片] 预期的图片文件名:`, expectedFileNames);
        
        // 检查文件上传进度的函数
        const checkUploadProgress = () => {
            // 重新查找容器（可能DOM已更新）
            flistContainer = findFlistContainer();
            if (!flistContainer) {
                console.warn('[PDF转图片] 文件列表容器已消失');
                return;
            }
            
            const flistItems = flistContainer.querySelectorAll('.flist-item');
            const currentFileCount = flistItems.length;
            const newFilesCount = currentFileCount - initialFileCount;
            
            // 详细的调试信息
            console.log(`[PDF转图片] 检查上传进度:`);
            console.log(`  - 初始文件数量: ${initialFileCount}`);
            console.log(`  - 当前文件数量: ${currentFileCount}`);
            console.log(`  - 新增文件数量: ${newFilesCount}`);
            console.log(`  - 期望新增数量: ${expectedImageCount}`);
            
            // 检查是否有预期的文件名出现（更可靠的检测方式）
            let foundExpectedFiles = 0;
            const foundFileNames = [];
            if (flistItems.length > 0 && expectedFileNames.length > 0) {
                flistItems.forEach((item) => {
                    const text = item.textContent.trim();
                    // 检查文件名是否匹配
                    expectedFileNames.forEach(expectedName => {
                        if (text.includes(expectedName)) {
                            foundExpectedFiles++;
                            foundFileNames.push(expectedName);
                        }
                    });
                });
            }
            console.log(`[PDF转图片] 找到预期的文件: ${foundExpectedFiles}/${expectedImageCount}`, foundFileNames);
            
            // 打印所有文件项，用于调试
            if (flistItems.length > 0) {
                console.log(`[PDF转图片] 当前文件列表:`);
                flistItems.forEach((item, index) => {
                    const text = item.textContent.trim();
                    console.log(`  [${index}] ${text.substring(0, 50)}...`);
                });
            }
            
            // 使用文件名匹配作为主要检测方式，文件数量作为备选
            const isUploadComplete = foundExpectedFiles >= expectedImageCount || newFilesCount >= expectedImageCount;
            
            // 更新上传进度提示
            if (!isUploadComplete) {
                // 优先显示文件名匹配的进度
                const progressCount = foundExpectedFiles > 0 ? foundExpectedFiles : newFilesCount;
                showLoading(`PDF转换完成，正在上传图片（${progressCount}/${expectedImageCount}），请稍候...`);
            } else {
                // 如果新增的文件数量达到期望值，或找到预期的文件，认为上传完成
                console.log(`[PDF转图片] 检测到上传完成条件满足，准备关闭提示`);
                const finalCount = foundExpectedFiles > 0 ? foundExpectedFiles : newFilesCount;
                completeUpload(finalCount, false);
            }
        };
        
        // 完成上传的处理函数
        const completeUpload = (uploadedCount, isTimeout) => {
            if (isCompleted) return;
            isCompleted = true;
            
            // 清理所有定时器和观察器
            if (timeoutId) {
                clearTimeout(timeoutId);
            }
            if (checkIntervalId) {
                clearInterval(checkIntervalId);
            }
            if (observer) {
                observer.disconnect();
            }
            
            console.log(`[PDF转图片] ${isTimeout ? '上传超时' : '上传完成'}，已上传 ${uploadedCount}/${expectedImageCount} 个文件`);
            
            hideLoading();
            if (uploadedCount >= expectedImageCount) {
                showUploadSuccessMessage(expectedImageCount);
            } else {
                showUploadSuccessMessage(uploadedCount, expectedImageCount);
            }
        };
        
        // 使用MutationObserver监听文件列表的变化（作为主要检测方式）
        observer = new MutationObserver((mutations) => {
            console.log(`[PDF转图片] MutationObserver检测到DOM变化，mutations数量: ${mutations.length}`);
            checkUploadProgress();
        });
        
        // 开始观察
        observer.observe(flistContainer, {
            childList: true,
            subtree: true,
            attributes: false,
            characterData: false
        });
        
        // 同时使用定期检查作为备选方案（每2秒检查一次）
        // 这样即使MutationObserver没触发，也能检测到文件上传完成
        checkIntervalId = setInterval(() => {
            checkUploadProgress();
        }, 2000);
        
        // 设置超时
        timeoutId = setTimeout(() => {
            const finalContainer = findFlistContainer();
            if (finalContainer) {
                const currentFileCount = finalContainer.querySelectorAll('.flist-item').length;
                const newFilesCount = currentFileCount - initialFileCount;
                completeUpload(newFilesCount, true);
            } else {
                completeUpload(0, true);
            }
        }, 60000); // 60秒超时
    }
    
    // 显示上传成功消息
    function showUploadSuccessMessage(uploadedCount, totalCount) {
        if (window.$ && window.$.messager) {
            let message = `已成功上传 ${uploadedCount} 张图片`;
            if (totalCount && uploadedCount < totalCount) {
                message += `（共 ${totalCount} 张，部分上传可能仍在进行中）`;
            }
            window.$.messager.show({
                title: '上传完成',
                msg: message,
                timeout: 3000,
                showType: 'slide'
            });
        }
    }

    // 更新上传按钮文本
    function updateUploadButtonText() {
        try {
            // 查找上传按钮，可能在 easyContainer 中
            // 尝试多种选择器，因为按钮可能在不同位置或不同结构
            const uploadButton = document.querySelector('#easyContainer .formBtn, #easyContainer a.formBtn, #easyContainer #btu_');
            if (uploadButton) {
                // 检查按钮文本元素（可能是 span.l-btn-text 或直接是按钮本身）
                let textElement = uploadButton.querySelector('.l-btn-text');
                if (!textElement) {
                    // 如果没有找到 .l-btn-text，尝试直接使用按钮本身
                    textElement = uploadButton;
                }
                
                if (textElement) {
                    const currentText = textElement.textContent.trim();
                    // 如果按钮文本是"上传"或者不包含"支持pdf"提示，就更新它
                    if (currentText === '上传' || (!currentText.includes('支持pdf') && !currentText.includes('支持PDF'))) {
                        textElement.textContent = '上传（支持pdf自动转图片）';
                        console.log('[PDF转图片] 已更新上传按钮文本');
                        return true;
                    }
                }
            }
            return false;
        } catch (error) {
            console.error('[PDF转图片] 更新按钮文本失败:', error);
            return false;
        }
    }

    // 处理文件选择
    async function handleFileSelection(event) {
        const fileInput = event.target;
        const files = Array.from(fileInput.files);
        
        if (files.length === 0) return;
        
        // 查找PDF文件
        const pdfFiles = files.filter(isPDFFile);
        if (pdfFiles.length === 0) {
            // 没有PDF文件，正常处理
            return;
        }
        
        // 阻止事件继续传播，我们将手动处理
        event.stopImmediatePropagation();
        event.preventDefault();
        
        console.log(`[PDF转图片] 检测到 ${pdfFiles.length} 个PDF文件`);
        console.log('[PDF转图片] PDF文件名:', pdfFiles.map(f => f.name));
        
        try {
            // 在开始转换前就显示加载提示
            console.log('[PDF转图片] 准备显示加载提示...');
            showLoading('正在将PDF转换为图片，请稍候...');
            console.log('[PDF转图片] 已调用showLoading');
            
            // 收集所有非PDF文件和转换后的图片文件
            const allFiles = files.filter(file => !isPDFFile(file));
            const convertedImages = [];
            
            // 转换每个PDF文件
            for (let fileIndex = 0; fileIndex < pdfFiles.length; fileIndex++) {
                const pdfFile = pdfFiles[fileIndex];
                const fileNumber = fileIndex + 1;
                const totalFiles = pdfFiles.length;
                
                // 更新提示：显示当前转换到第几个文件
                if (totalFiles > 1) {
                    showLoading(`正在将PDF转换为图片（${fileNumber}/${totalFiles}）：${pdfFile.name}...`);
                } else {
                    showLoading(`正在将PDF转换为图片：${pdfFile.name}...`);
                }
                
                // 转换PDF文件，传入进度回调
                try {
                    const images = await convertPDFToImages(pdfFile, (currentIndex, totalSelected, pageNum) => {
                        // 更新提示：显示当前转换到第几页
                        if (totalFiles > 1) {
                            showLoading(`正在将PDF转换为图片（${fileNumber}/${totalFiles}）：${pdfFile.name}，正在转换第 ${currentIndex}/${totalSelected} 页（原PDF第 ${pageNum} 页）...`);
                        } else {
                            if (totalSelected > 1) {
                                showLoading(`正在将PDF转换为图片：${pdfFile.name}，正在转换第 ${currentIndex}/${totalSelected} 页（原PDF第 ${pageNum} 页）...`);
                            } else {
                                showLoading(`正在将PDF转换为图片：${pdfFile.name}，正在转换第 ${pageNum} 页...`);
                            }
                        }
                    });
                    convertedImages.push(...images);
                } catch (cancelError) {
                    // 如果用户取消了页面选择，停止处理后续文件
                    if (cancelError.message === '用户取消') {
                        console.log('[PDF转图片] 用户取消了页面选择');
                        hideLoading();
                        // 清空文件输入框
                        fileInput.value = '';
                        return;
                    }
                    // 其他错误继续抛出
                    throw cancelError;
                }
            }
            
            // 如果没有转换任何图片（可能用户都取消了），直接返回
            if (convertedImages.length === 0 && pdfFiles.length > 0) {
                console.log('[PDF转图片] 没有转换任何图片，可能是用户取消了所有文件');
                hideLoading();
                fileInput.value = '';
                return;
            }
            
            // 合并所有文件
            allFiles.push(...convertedImages);
            
            // 创建新的FileList对象
            const dataTransfer = new DataTransfer();
            allFiles.forEach(file => {
                dataTransfer.items.add(file);
            });
            
            // 更新文件输入
            fileInput.files = dataTransfer.files;
            
            console.log(`[PDF转图片] 已转换 ${pdfFiles.length} 个PDF文件为 ${convertedImages.length} 张图片`);
            
            // 触发change事件，让easyUpload插件处理转换后的图片
            // 使用setTimeout确保DOM更新完成
            setTimeout(() => {
                const changeEvent = new Event('change', { bubbles: true, cancelable: true });
                fileInput.dispatchEvent(changeEvent);
                
                // 转换完成，关闭加载提示
                hideLoading();
                
                // 显示成功提示
                if (window.$ && window.$.messager) {
                    window.$.messager.show({
                        title: 'PDF转换完成',
                        msg: `已将 ${pdfFiles.length} 个PDF文件转换为 ${convertedImages.length} 张图片，正在上传中...`,
                        timeout: 3000,
                        showType: 'slide'
                    });
                }
            }, 100);
            
        } catch (error) {
            console.error('[PDF转图片] 处理失败:', error);
            hideLoading(); // 出错时立即隐藏
            
            // 如果用户取消，不清空文件输入框，也不显示错误
            if (error.message === '用户取消') {
                console.log('[PDF转图片] 用户取消了转换');
                fileInput.value = '';
                return;
            }
            
            // 其他错误显示提示
            if (window.$ && window.$.messager) {
                window.$.messager.alert('错误', 'PDF转换失败: ' + error.message, 'error');
            } else {
                alert('PDF转换失败: ' + error.message);
            }
            
            // 清空文件输入框
            fileInput.value = '';
        }
    }

    // 初始化：监听文件选择事件
    function init() {
        // 等待页面加载完成
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', init);
            return;
        }
        
        // 检查是否是明细上传页面
        if (!isDetailUploadPage()) {
            console.log('[PDF转图片] 不是明细上传页面，跳过');
            return;
        }
        
        console.log('[PDF转图片] 初始化PDF转图片功能');
        
        // 更新上传按钮文本
        updateUploadButtonText();
        
        // 如果按钮还没出现，等待一段时间后重试
        setTimeout(() => {
            updateUploadButtonText();
        }, 500);
        
        // 查找文件输入框（可能有多个，比如在easyContainer中）
        const fileInputs = document.querySelectorAll('#easyContainer input[type="file"]');
        
        if (fileInputs.length === 0) {
            console.warn('[PDF转图片] 未找到文件输入框');
            // 等待一段时间后重试（可能easyUpload插件还没初始化）
            setTimeout(() => {
                const retryInputs = document.querySelectorAll('#easyContainer input[type="file"]');
                if (retryInputs.length > 0) {
                    attachListeners(retryInputs);
                }
                // 再次尝试更新按钮文本
                updateUploadButtonText();
            }, 1000);
            return;
        }
        
        attachListeners(fileInputs);
    }

    // 为文件输入框绑定事件
    function attachListeners(fileInputs) {
        fileInputs.forEach(input => {
            // 检查是否已经绑定过
            if (input.hasAttribute('data-pdf-converter-attached')) {
                return;
            }
            
            // 标记已绑定
            input.setAttribute('data-pdf-converter-attached', 'true');
            
            // 使用捕获阶段，确保我们的处理在其他监听器之前执行
            input.addEventListener('change', handleFileSelection, true);
            console.log('[PDF转图片] 已绑定文件选择监听器');
        });
    }

    // 启动
    init();
    
    // 如果easyUpload插件是延迟加载的，使用MutationObserver监听DOM变化
    const observer = new MutationObserver((mutations) => {
        // 检查是否是明细上传页面
        if (!isDetailUploadPage()) {
            return;
        }
        
        // 尝试更新按钮文本
        updateUploadButtonText();
        
        const fileInputs = document.querySelectorAll('#easyContainer input[type="file"]');
        if (fileInputs.length > 0) {
            fileInputs.forEach(input => {
                // 检查是否已经绑定过监听器
                if (!input.hasAttribute('data-pdf-converter-attached')) {
                    input.setAttribute('data-pdf-converter-attached', 'true');
                    // 使用捕获阶段，确保我们的处理在其他监听器之前执行
                    input.addEventListener('change', handleFileSelection, true);
                    console.log('[PDF转图片] 检测到新的文件输入框，已绑定监听器');
                }
            });
        }
    });
    
    // 观察easyContainer的变化
    const easyContainer = document.querySelector('#easyContainer');
    if (easyContainer) {
        observer.observe(easyContainer, {
            childList: true,
            subtree: true
        });
    }

})();

