// 通用表单处理类 - 基于配置动态生成表单功能
(function() {
    'use strict';

    /**
     * 通用表单类 - 基于配置动态处理所有表单
     */
    class GenericForm extends BaseForm {
        constructor(formKey, formConfig) {
            super(formKey, formConfig.name);
            this.formKey = formKey;
            this.config = formConfig;
        }

        /**
         * 获取主文档（如果不在iframe中，返回当前文档）
         */
        getMainDocument(doc = null) {
            try {
                if (doc) return doc;
                if (window.top && window.top !== window.self) {
                    return window.top.document;
                }
                return document;
            } catch (e) {
                console.log('无法访问主文档，使用当前文档:', e.message);
                return document;
            }
        }

        /**
         * 检查是否是此表单类型
         */
        isFormType(doc = null) {
            let targetDoc = doc;
            if (!targetDoc) {
                targetDoc = this.getMainDocument();
            }
            
            console.log(`=== GenericForm.isFormType 调试信息 (${this.config.name}) ===`);
            console.log('检查文档:', targetDoc === document ? '当前文档' : (targetDoc === window.top?.document ? '主文档' : '其他文档'));
            
            // 查找当前选中的 tab
            const selectedTab = targetDoc.querySelector('.tabs-selected');
            console.log('selectedTab:', selectedTab);
            
            // 如果找不到选中的tab，尝试查找所有tabs看看结构
            if (!selectedTab) {
                const allTabs = targetDoc.querySelectorAll('.tabs li');
                console.log('未找到选中的 tab，尝试查找所有 tabs');
                console.log('所有 tabs 数量:', allTabs.length);
                allTabs.forEach((tab, index) => {
                    const title = tab.querySelector('.tabs-title');
                    console.log(`Tab ${index}: class="${tab.className}", title="${title ? title.textContent.trim() : 'null'}"`);
                });
                console.log('=== 调试信息结束 ===');
                return false;
            }
            
            // 查找选中 tab 中的标题元素
            const tabTitle = selectedTab.querySelector('.tabs-title');
            console.log('tabTitle:', tabTitle);
            console.log('tabTitle.textContent:', tabTitle ? tabTitle.textContent : 'null');
            console.log('tabTitle.textContent.trim():', tabTitle ? tabTitle.textContent.trim() : 'null');
            
            if (!tabTitle) {
                console.log('未找到 tabTitle');
                console.log('=== 调试信息结束 ===');
                return false;
            }
            
            // 检查标题文本匹配
            const titleText = tabTitle.textContent.trim();
            
            // 精确匹配
            if (this.config.tabTitle && titleText === this.config.tabTitle) {
                console.log('精确匹配成功:', titleText);
                console.log('=== 调试信息结束 ===');
                return true;
            }
            
            // 关键词匹配
            if (this.config.tabTitleKeywords && this.config.tabTitleKeywords.length > 0) {
                for (const keyword of this.config.tabTitleKeywords) {
                    if (titleText.includes(keyword)) {
                        if (!titleText.includes('创建') && !titleText.includes('编辑')) {
                            console.log('tab标题未包含“创建”或“编辑”，判定失败');
                            continue;
                        }
                        console.log(`关键词匹配成功: "${keyword}" in "${titleText}"`);
                        console.log('=== 调试信息结束 ===');
                        return true;
                    }
                }
            }
            
            console.log('匹配失败');
            console.log('期望文本:', this.config.tabTitle);
            console.log('期望关键词:', this.config.tabTitleKeywords);
            console.log('实际文本:', `"${titleText}"`);
            console.log('=== 调试信息结束 ===');
            return false;
        }

        /**
         * 直接填充人员字段
         */
        directFillPersonField(inputId, hiddenId, userIds, userNames, doc = document) {
            try {
                const input = doc.querySelector(`#${inputId}`);
                const hidden = doc.querySelector(`#${hiddenId}`);
                
                if (!input || !hidden) {
                    console.log(`未找到字段: ${inputId} / ${hiddenId}`);
                    return false;
                }
                
                console.log(`开始直接填充 ${inputId}`);
                console.log(`原值: ${input.value}`);
                console.log(`新值: ${userNames}`);
                
                // 先清空字段值，确保覆盖
                input.value = '';
                hidden.value = '';
                
                // 清空时触发事件
                const clearEvents = ['change', 'input'];
                clearEvents.forEach(eventType => {
                    const event = new Event(eventType, { bubbles: true, cancelable: true });
                    input.dispatchEvent(event);
                    hidden.dispatchEvent(event);
                });
                
                // 设置新值
                input.value = userNames;
                hidden.value = userIds;
                
                const events = ['change', 'input', 'blur', 'keyup'];
                events.forEach(eventType => {
                    const event = new Event(eventType, { bubbles: true, cancelable: true });
                    input.dispatchEvent(event);
                    hidden.dispatchEvent(event);
                });
                
                if (window.$ && typeof $.fn.val === 'function') {
                    try {
                        $(input).val(userNames).trigger('change');
                        $(hidden).val(userIds).trigger('change');
                        
                        if (window.$.fn.validatebox) {
                            $(input).validatebox('validate');
                        }
                    } catch (e) {
                        console.log(`jQuery 更新失败 (${inputId}):`, e.message);
                    }
                }
                
                console.log(`完成直接填充 ${inputId}`);
                return true;
                
            } catch (error) {
                console.error(`直接填充 ${inputId} 失败:`, error);
                return false;
            }
        }

        /**
         * 直接填充部门字段
         */
        directFillDeptField(inputId, hiddenId, deptId, deptName, doc = document) {
            try {
                const input = doc.querySelector(`#${inputId}`);
                const hidden = doc.querySelector(`#${hiddenId}`);
                
                if (!input || !hidden) {
                    console.log(`未找到字段: ${inputId} / ${hiddenId}`);
                    return false;
                }
                
                console.log(`开始直接填充部门 ${inputId}`);
                console.log(`原值: ${input.value}`);
                console.log(`新值: ${deptName}`);
                
                // 先清空字段值，确保覆盖
                input.value = '';
                hidden.value = '';
                
                // 清空时触发事件
                const clearEvents = ['change', 'input'];
                clearEvents.forEach(eventType => {
                    const event = new Event(eventType, { bubbles: true, cancelable: true });
                    input.dispatchEvent(event);
                    hidden.dispatchEvent(event);
                });
                
                // 设置新值
                input.value = deptName;
                hidden.value = deptId;
                
                const events = ['change', 'input', 'blur', 'keyup'];
                events.forEach(eventType => {
                    const event = new Event(eventType, { bubbles: true, cancelable: true });
                    input.dispatchEvent(event);
                    hidden.dispatchEvent(event);
                });
                
                if (window.$ && typeof $.fn.val === 'function') {
                    try {
                        $(input).val(deptName).trigger('change');
                        $(hidden).val(deptId).trigger('change');
                        
                        if (window.$.fn.validatebox) {
                            $(input).validatebox('validate');
                        }
                    } catch (e) {
                        console.log(`jQuery 更新失败 (${inputId}):`, e.message);
                    }
                }
                
                console.log(`完成直接填充部门 ${inputId}`);
                return true;
                
            } catch (error) {
                console.error(`直接填充部门 ${inputId} 失败:`, error);
                return false;
            }
        }

        /**
         * 自动填充表单
         */
        autoFill(config = {}, doc = null) {
            try {
                console.log(`=== 开始自动填充${this.config.name} ===`);
                
                // 检测表单类型时使用主文档（tab在主文档中）
                const mainDoc = this.getMainDocument();
                if (!this.isFormType(mainDoc)) {
                    console.log(`当前页面不是${this.config.name}`);
                    return false;
                }
                
                // 填充表单字段时使用传入的文档，如果没有则使用当前文档（可能在iframe中）
                const targetDoc = doc || document;
                
                // 获取配置，优先使用传入的配置，然后是存储的配置，最后是默认配置
                const formConfigKey = `${this.formKey}FormDefaults`;
                const defaults = config[formConfigKey] || config.forms?.[this.formKey] || this.getDefaultConfig();
                
                let successCount = 0;
                const totalFields = this.config.fields.length;
                
                // 遍历配置的字段进行填充
                this.config.fields.forEach(field => {
                    let success = false;
                    
                    if (field.type === 'person') {
                        // 人员字段 - 优先使用保存的配置，即使为空也要使用
                        const userIds = defaults[`${field.inputId.replace('Name', 'Id').replace('name', 'id')}`];
                        const userNames = defaults[field.inputId];
                        
                        // 如果保存的配置中有值（包括空字符串），就使用保存的配置
                        if (defaults.hasOwnProperty(`${field.inputId.replace('Name', 'Id').replace('name', 'id')}`) && 
                            defaults.hasOwnProperty(field.inputId)) {
                            success = this.directFillPersonField(field.inputId, field.hiddenId, userIds || '', userNames || '', targetDoc);
                        } else if (userIds && userNames) {
                            // 如果保存的配置中没有这些字段，才使用默认配置
                            success = this.directFillPersonField(field.inputId, field.hiddenId, userIds, userNames, targetDoc);
                        }
                    } else if (field.type === 'dept') {
                        // 部门字段 - 优先使用保存的配置，即使为空也要使用
                        const deptId = defaults[field.hiddenId];
                        const deptName = defaults[field.inputId];
                        
                        // 如果保存的配置中有值（包括空字符串），就使用保存的配置
                        if (defaults.hasOwnProperty(field.hiddenId) && defaults.hasOwnProperty(field.inputId)) {
                            success = this.directFillDeptField(field.inputId, field.hiddenId, deptId || '', deptName || '', targetDoc);
                        } else if (deptId && deptName) {
                            // 如果保存的配置中没有这些字段，才使用默认配置
                            success = this.directFillDeptField(field.inputId, field.hiddenId, deptId, deptName, targetDoc);
                        }
                    }
                    
                    if (success) {
                        successCount++;
                    }
                });
                
                console.log(`${this.config.name}填充完成: ${successCount}/${totalFields} 个字段已填充`);
                
                // 延迟检查填充结果
                setTimeout(() => {
                    console.log('=== 填充后状态检查 ===');
                    this.config.fields.forEach(field => {
                        const hiddenEl = targetDoc.querySelector(`#${field.hiddenId}`);
                        const visibleEl = targetDoc.querySelector(`#${field.inputId}`);
                        console.log(`${field.label}:`);
                        if (hiddenEl) {
                            console.log(`  - 隐藏字段值: ${hiddenEl.value}`);
                        }
                        console.log(`  - 显示字段值: ${visibleEl ? visibleEl.value : '未找到'}`);
                    });
                }, 200);
                
                console.log(`=== ${this.config.name}自动填充完成 ===`);
                return successCount > 0;
                
            } catch (error) {
                console.error(`自动填充${this.config.name}失败:`, error);
                return false;
            }
        }

        /**
         * 从当前页面提取配置
         */
        captureConfig(doc = null) {
            try {
                // 检测表单类型时使用主文档（tab在主文档中）
                const mainDoc = this.getMainDocument();
                if (!this.isFormType(mainDoc)) {
                    console.log(`当前页面不是${this.config.name}，无法提取配置`);
                    return null;
                }
                
                // 提取配置时使用传入的文档，如果没有则使用当前文档（可能在iframe中）
                const targetDoc = doc || document;
                
                function getInputValue(selector, doc) {
                    const el = doc.querySelector(selector);
                    if (!el) return '';
                    
                    if (el.value && el.value.trim() !== '') {
                        return el.value.trim();
                    }
                    
                    if (window.$ && typeof $.fn.val === 'function') {
                        try {
                            const jqVal = $(el).val();
                            if (jqVal && jqVal.trim() !== '') {
                                return jqVal.trim();
                            }
                        } catch (e) {
                            // jQuery 获取失败，继续尝试其他方法
                        }
                    }
                    
                    if (el.textContent && el.textContent.trim() !== '') {
                        return el.textContent.trim();
                    }
                    
                    return '';
                }
                
                const config = {};
                
                // 根据字段配置提取值
                this.config.fields.forEach(field => {
                    const hiddenValue = targetDoc.querySelector(`#${field.hiddenId}`)?.value || '';
                    const visibleValue = getInputValue(`#${field.inputId}`, targetDoc);
                    
                    config[field.hiddenId] = hiddenValue;
                    config[field.inputId] = visibleValue;
                });
                
                console.log(`=== 提取的${this.config.name}配置 ===`);
                console.log(config);
                
                // 即使所有字段都为空，也要返回配置对象（用于覆盖原有配置）
                const hasData = Object.values(config).some(v => v && v.trim() !== '');
                if (!hasData) {
                    console.log('所有字段都为空，但仍返回空配置用于覆盖原有配置');
                }
                
                return config;
                
            } catch (error) {
                console.error(`提取${this.config.name}配置失败:`, error);
                return null;
            }
        }

        /**
         * 获取默认配置
         */
        getDefaultConfig() {
            const config = {};
            
            this.config.fields.forEach(field => {
                if (field.type === 'person') {
                    config[field.hiddenId] = field.defaultIds;
                    config[field.inputId] = field.defaultNames;
                } else if (field.type === 'dept') {
                    config[field.hiddenId] = field.defaultId;
                    config[field.inputId] = field.defaultName;
                }
            });
            
            return config;
        }

        /**
         * 获取配置字段列表
         */
        getConfigFields() {
            return this.config.fields.map(field => ({
                key: field.inputId,
                label: field.label,
                type: 'text',
                placeholder: field.type === 'person' ? field.defaultNames : field.defaultName,
                hiddenKey: field.hiddenId,
                hiddenLabel: `${field.label}ID`,
                hiddenPlaceholder: field.type === 'person' ? field.defaultIds : field.defaultId
            }));
        }
    }

    // 导出到全局
    window.GenericForm = GenericForm;

})();
