// 表单基类 - 定义统一的表单接口
(function() {
    'use strict';

    /**
     * 表单基类
     * 所有表单实现都应该继承或实现这些方法
     */
    class BaseForm {
        /**
         * 构造函数
         * @param {string} formName - 表单名称（如 'finance', 'purchase-request'）
         * @param {string} displayName - 表单显示名称（如 '财务审批表'）
         */
        constructor(formName, displayName) {
            this.formName = formName;
            this.displayName = displayName;
        }

        /**
         * 获取表单名称
         * @returns {string} 表单名称
         */
        getFormName() {
            return this.formName;
        }

        /**
         * 获取表单显示名称
         * @returns {string} 表单显示名称
         */
        getDisplayName() {
            return this.displayName;
        }

        /**
         * 检查当前页面是否是此表单类型
         * @param {Document} doc - 要检查的文档对象（可选，默认为当前文档）
         * @returns {boolean} 是否是此表单类型
         */
        isFormType(doc = null) {
            throw new Error('isFormType() 方法必须在子类中实现');
        }

        /**
         * 自动填充表单
         * @param {Object} config - 表单配置对象
         * @param {Document} doc - 要填充的文档对象（可选，默认为当前文档）
         * @returns {boolean} 是否填充成功
         */
        autoFill(config = {}, doc = null) {
            throw new Error('autoFill() 方法必须在子类中实现');
        }

        /**
         * 从当前页面提取表单配置
         * @param {Document} doc - 要提取的文档对象（可选，默认为当前文档）
         * @returns {Object|null} 配置对象，如果无法提取则返回null
         */
        captureConfig(doc = null) {
            throw new Error('captureConfig() 方法必须在子类中实现');
        }

        /**
         * 获取表单的默认配置
         * @returns {Object} 默认配置对象
         */
        getDefaultConfig() {
            return {};
        }

        /**
         * 获取表单配置的字段列表（用于配置界面）
         * @returns {Array} 字段配置数组，每个元素包含 {key, label, type, placeholder}
         */
        getConfigFields() {
            return [];
        }
    }

    // 导出到全局
    window.BaseForm = BaseForm;

})();
