// OA审批系统自动填写助手 - Content Script (主入口)
(function() {
    'use strict';

    // 表单管理器 - 管理所有已注册的表单
    class FormManager {
        constructor() {
            this.forms = new Map(); // 存储所有表单实例
            this.currentForm = null; // 当前检测到的表单
        }

        /**
         * 注册表单
         * @param {BaseForm} formInstance - 表单实例
         */
        registerForm(formInstance) {
            if (!formInstance || typeof formInstance.isFormType !== 'function') {
                console.error('无效的表单实例:', formInstance);
                return;
            }
            this.forms.set(formInstance.getFormName(), formInstance);
            console.log(`已注册表单: ${formInstance.getDisplayName()} (${formInstance.getFormName()})`);
        }

        /**
         * 检测当前页面的表单类型
         * @param {Document} doc - 要检测的文档对象（可选）
         * @returns {BaseForm|null} 匹配的表单实例，如果没有匹配则返回null
         */
        detectFormType(doc = null) {
            for (const [formName, form] of this.forms) {
                try {
                    if (form.isFormType(doc)) {
                        console.log(`检测到表单类型: ${form.getDisplayName()} (${formName})`);
                        return form;
                    }
                } catch (error) {
                    console.error(`检测表单 ${formName} 时出错:`, error);
                }
            }
            return null;
        }

        /**
         * 获取当前表单
         * @returns {BaseForm|null}
         */
        getCurrentForm() {
            if (!this.currentForm) {
                this.currentForm = this.detectFormType();
            }
            return this.currentForm;
        }

        /**
         * 刷新当前表单（重新检测）
         */
        refreshCurrentForm() {
            this.currentForm = this.detectFormType();
            return this.currentForm;
        }

        /**
         * 获取所有已注册的表单列表
         * @returns {Array} 表单列表
         */
        getAllForms() {
            return Array.from(this.forms.values());
        }

        /**
         * 根据表单名称获取表单实例
         * @param {string} formName - 表单名称
         * @returns {BaseForm|null}
         */
        getFormByName(formName) {
            return this.forms.get(formName) || null;
        }
    }

    // 全局表单管理器实例
    const formManager = new FormManager();

    // 配置信息 - 从chrome.storage读取
    let CONFIG = {
        // 各表单的默认配置
        forms: {}
    };

    // 加载配置
    function loadConfig() {
        chrome.storage.sync.get(['oaConfig'], function(result) {
            if (result.oaConfig) {
                // 合并配置，保持向后兼容
                const mergedConfig = {};
                // forms配置从存储中读取，如果不存在则为空对象（默认配置从form子类获取）
                if (result.oaConfig.forms) {
                    mergedConfig.forms = {...result.oaConfig.forms};
                } else {
                    // 向后兼容：如果存在 financeFormDefaults，转换为新格式
                    mergedConfig.forms = {};
                    if (result.oaConfig.financeFormDefaults) {
                        mergedConfig.forms.finance = result.oaConfig.financeFormDefaults;
                    }
                }
                CONFIG = mergedConfig;
                console.log('已加载配置:', CONFIG);
                if (CONFIG.forms) {
                    Object.keys(CONFIG.forms).forEach(formName => {
                        console.log(`  - ${formName}:`, CONFIG.forms[formName]);
                    });
                }
            } else {
                CONFIG = {forms: {}};
                console.log('使用默认配置:', CONFIG);
            }
        });
    }

    // 获取主文档（如果不在iframe中，返回当前文档）
    function getMainDocument() {
        try {
            if (window.top && window.top !== window.self) {
                return window.top.document;
            }
            return document;
        } catch (e) {
            console.log('无法访问主文档，使用当前文档:', e.message);
            return document;
        }
    }

    // 等待函数
    function sleep(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    // 轮询等待元素出现
    async function waitForElementPolling(selector, options = {}) {
        const { interval = 50, maxWait = 5000, doc = null } = options;
        const targetDoc = doc || document;
        const startTime = Date.now();
        
        while (Date.now() - startTime < maxWait) {
            const element = targetDoc.querySelector(selector);
            if (element) {
                return element;
            }
            await sleep(interval);
        }
        return null;
    }

    // 将表单名称转换为驼峰命名的配置key（用于向后兼容）
    function getFormDefaultsKey(formName) {
        // 将 'purchase-request' 转换为 'purchaseRequest'
        return formName.replace(/-([a-z])/g, (match, letter) => letter.toUpperCase()) + 'FormDefaults';
    }

    // 自动填充当前表单
    function autoFillCurrentForm() {
        const currentForm = formManager.getCurrentForm();
        if (!currentForm) {
            console.log('未检测到支持的表单类型');
            return false;
        }

        const formName = currentForm.getFormName();
        const formConfig = CONFIG.forms[formName] || currentForm.getDefaultConfig();
        const configKey = getFormDefaultsKey(formName);
        
        console.log(`=== 自动填充配置信息 ===`);
        console.log(`表单名称: ${formName}`);
        console.log(`配置Key: ${configKey}`);
        console.log(`是否使用存储的配置: ${!!CONFIG.forms[formName]}`);
        console.log(`配置内容:`, formConfig);
        
        return currentForm.autoFill({
            ...CONFIG,
            [configKey]: formConfig  // 保持向后兼容的命名
        });
    }

    // 从当前页面提取配置
    function captureCurrentFormConfig() {
        const currentForm = formManager.getCurrentForm();
        if (!currentForm) {
            console.log('未检测到支持的表单类型，无法提取配置');
            return null;
        }

        return currentForm.captureConfig();
    }

    // 保存提取的配置
    function saveCapturedConfig() {
        const currentForm = formManager.getCurrentForm();
        if (!currentForm) {
            alert('未检测到支持的表单类型');
            return;
        }

        const captured = captureCurrentFormConfig();
        if (captured === null) {
            alert('未能提取配置信息，请确保页面已填写完整');
            return;
        }
        
        // 即使captured为空对象（所有字段都为空），也允许保存

        const formName = currentForm.getFormName();
        
        // 更新 CONFIG
        if (!CONFIG.forms[formName]) {
            CONFIG.forms[formName] = {};
        }
        CONFIG.forms[formName] = {
            ...CONFIG.forms[formName],
            ...captured
        };

        // 保存到 chrome.storage
        chrome.storage.sync.get(['oaConfig'], function(result) {
            const existingConfig = result.oaConfig || {};
            const updatedConfig = {
                ...existingConfig,
                forms: {
                    ...(existingConfig.forms || {}),
                    [formName]: CONFIG.forms[formName]
                }
            };

            chrome.storage.sync.set({oaConfig: updatedConfig}, function() {
                console.log('已保存提取的配置:', updatedConfig);
                console.log(`表单 ${formName} 的配置已保存:`, CONFIG.forms[formName]);
                
                const configStr = JSON.stringify(captured, null, 2).replace(/"/g, '');
                const hasData = Object.values(captured).some(v => v && v.trim() !== '');
                const message = hasData 
                    ? `已成功提取并保存${currentForm.getDisplayName()}配置！\n\n提取的信息：\n${configStr}`
                    : `已成功保存${currentForm.getDisplayName()}配置！\n\n注意：所有字段都为空，已清空原有配置。`;
                alert(message);
            });
        });
    }

    // 显示确认对话框（已改为直接执行）
    function showConfirmDialog() {
        const currentForm = formManager.getCurrentForm();
        if (!currentForm) {
            alert('未检测到支持的表单类型');
            return;
        }

        // 直接执行，不再弹出确认对话框
        autoFillCurrentForm();
    }

    // 查看已保存的配置
    function viewSavedConfig() {
        const currentForm = formManager.getCurrentForm();
        if (!currentForm) {
            alert('未检测到支持的表单类型');
            return;
        }

        const formName = currentForm.getFormName();
        
        // 从CONFIG中获取当前表单的配置
        const savedConfig = CONFIG.forms[formName] || currentForm.getDefaultConfig();
        
        if (!savedConfig || Object.keys(savedConfig).length === 0) {
            alert(`当前页面尚未保存任何${currentForm.getDisplayName()}配置。`);
            return;
        }

        // 构建显示内容 - 使用可滚动的HTML弹窗
        let configText = `<div style="max-height: 500px; overflow-y: auto; text-align: left; font-family: monospace; white-space: pre-wrap;">`;
        configText += `<h3 style="margin: 0 0 10px 0; color: #333;">当前${currentForm.getDisplayName()}的保存配置：</h3><hr style="margin: 10px 0;">`;
        
        // 遍历表单字段，将配置转换为可读格式
        currentForm.config.fields.forEach(field => {
            const visibleValue = savedConfig[field.inputId] || '(未设置)';
            const hiddenValue = savedConfig[field.hiddenId] || '(未设置)';
            
            configText += `<strong>${field.label}:</strong>\n`;
            configText += `  显示值: ${visibleValue}\n`;
            configText += `  ID值: ${hiddenValue}\n\n`;
        });
        
        configText += `<hr style="margin: 10px 0;">`;
        configText += `<strong>原始配置数据：</strong>\n`;
        configText += JSON.stringify(savedConfig, null, 2);
        configText += `</div>`;
        
        // 创建自定义弹窗
        const modal = document.createElement('div');
        modal.style.cssText = 'position:fixed;top:0;left:0;width:100%;height:100%;background:rgba(0,0,0,0.5);z-index:10000;display:flex;align-items:center;justify-content:center;';
        
        const content = document.createElement('div');
        content.style.cssText = 'background:white;padding:20px;border-radius:8px;max-width:600px;max-height:80vh;overflow-y:auto;box-shadow:0 4px 6px rgba(0,0,0,0.1);';
        content.innerHTML = configText;
        
        const closeBtn = document.createElement('button');
        closeBtn.textContent = '关闭';
        closeBtn.style.cssText = 'margin-top:15px;padding:8px 16px;background:#007cba;color:white;border:none;border-radius:4px;cursor:pointer;width:100%;';
        closeBtn.onclick = () => modal.remove();
        
        content.appendChild(closeBtn);
        modal.appendChild(content);
        
        // 点击外部关闭
        modal.onclick = (e) => {
            if (e.target === modal) {
                modal.remove();
            }
        };
        
        document.body.appendChild(modal);
    }

    // 添加手动触发按钮
    async function addManualTrigger() {
        // 检查是否已经添加了按钮
        if (document.querySelector('#oa-auto-fill-btn')) {
            return;
        }

        const mainDoc = getMainDocument();

        // 等待 tab 元素出现
        const tabsContainer = await waitForElementPolling('.tabs', { maxWait: 2000, doc: mainDoc });
        if (!tabsContainer) {
            console.log('未找到 tabs 容器，可能不是正确的页面');
            return;
        }

        // 等待选中的 tab 和标题元素出现
        await waitForElementPolling('.tabs-selected', { maxWait: 2000, doc: mainDoc });
        await sleep(300); // 额外等待确保DOM完全渲染

        // 检测当前表单类型（重试几次，因为DOM可能还在加载中）
        let currentForm = null;
        for (let i = 0; i < 3; i++) {
            currentForm = formManager.refreshCurrentForm();
            if (currentForm) {
                break;
            }
            console.log(`表单检测尝试 ${i + 1}/3，等待重试...`);
            await sleep(500);
        }

        if (!currentForm) {
            console.log('当前页面不是支持的表单类型，跳过按钮添加');
            console.log('尝试调试：检查所有tab标题');
            const allTabs = mainDoc.querySelectorAll('.tabs li');
            allTabs.forEach((tab, index) => {
                const title = tab.querySelector('.tabs-title');
                console.log(`Tab ${index}: class="${tab.className}", title="${title ? title.textContent.trim() : 'null'}"`);
            });
            return;
        }

        // 添加自动填充按钮
        const button = document.createElement('button');
        button.id = 'oa-auto-fill-btn';
        button.textContent = `自动填充${currentForm.getDisplayName()}`;
        button.style.cssText = 'position:fixed;top:10px;right:10px;z-index:9999;background:#007cba;color:white;border:none;padding:8px 12px;border-radius:4px;cursor:pointer;font-size:12px;';
        button.onclick = showConfirmDialog;
        document.body.appendChild(button);

        // 添加提取配置按钮
        const captureButton = document.createElement('button');
        captureButton.id = 'oa-capture-btn';
        captureButton.textContent = '保存页面配置';
        captureButton.style.cssText = 'position:fixed;top:90px;right:10px;z-index:9999;background:#ffc107;color:black;border:none;padding:8px 12px;border-radius:4px;cursor:pointer;font-size:12px;';
        captureButton.onclick = () => {
            // 直接执行，不再弹出确认对话框
            saveCapturedConfig();
        };
        document.body.appendChild(captureButton);

        // 添加查看配置按钮
        const viewConfigButton = document.createElement('button');
        viewConfigButton.id = 'oa-view-config-btn';
        viewConfigButton.textContent = '查看当前配置';
        viewConfigButton.style.cssText = 'position:fixed;top:50px;right:10px;z-index:9999;background:#28a745;color:white;border:none;padding:8px 12px;border-radius:4px;cursor:pointer;font-size:12px;';
        viewConfigButton.onclick = () => {
            viewSavedConfig();
        };
        document.body.appendChild(viewConfigButton);

        console.log(`已在${currentForm.getDisplayName()}页面添加按钮`);
    }

    // 监听iframe内容变化
    function observeIframes() {
        // 暂不实现iframe监听
    }

    // 初始化：注册所有表单
    function initializeForms() {
        // 等待BaseForm类加载完成
        if (typeof BaseForm === 'undefined') {
            console.error('BaseForm类未加载');
            return;
        }

        // 等待GenericForm类和FORM_CONFIGS加载完成
        if (typeof GenericForm === 'undefined' || typeof FORM_CONFIGS === 'undefined') {
            console.error('GenericForm类或FORM_CONFIGS未加载');
            return;
        }

        // 使用通用表单系统注册所有表单
        Object.keys(FORM_CONFIGS).forEach(formKey => {
            const formConfig = FORM_CONFIGS[formKey];
            const formInstance = new GenericForm(formKey, formConfig);
            formManager.registerForm(formInstance);
        });

        console.log('通用表单系统初始化完成，已注册', formManager.getAllForms().length, '个表单');
    }


    // 页面加载完成后初始化
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            loadConfig();
            setTimeout(() => {
                initializeForms();
                addManualTrigger();
                observeIframes();
            }, 500);
        });
    } else {
        loadConfig();
        setTimeout(() => {
            initializeForms();
            addManualTrigger();
            observeIframes();
        }, 500);
    }

    // 导出到全局（供调试使用）
    window.OAFormManager = formManager;
    window.OAConfig = CONFIG;

})();